/**
 * @file GenericCapture.h
 * This header file defines the GenericCapture class.
 * @author Tomas Urban
 * @version 0.2
 * @date 21/07/2009
 */
#ifndef GENERIC_CAPTURE_H
#define GENERIC_CAPTURE_H

#include "Messages/CapturedData.h"
#include "CaptureDispatch.h"


/**
 * Abstract class describing a capturing device. It defines the common interface and several
 * useful methods. However, the capturing code itself must be implemented in child classes
 * according to the means used for capture (strategy design pattern).
 */
class GenericCapture
{
private:
	CaptureDispatch * m_pDispatch;
	struct timeval m_offset;
	struct timeval m_startPoint;
	struct timeval m_timestamp;
	void UpdateTimestamp();
protected:
	GenericCapture();
	/**
	 * This method is used for processing the captured data, passing it to an associated
	 * CaptureDispatch object.
	 * @param pData Data to process
	 */
	void ProcessCapturedData(CapturedData * pData);
	/**
	 * Sets the capture start point timestamp which is used in time-based filtering
	 * @param timestamp Timestamp of the beginning of capture
	 */
	void SetStartPoint(struct timeval timestamp);
public:	
	virtual ~GenericCapture(void);
	/**
	 * Sets the object used for delivering captured data to the final recipient.
	 */
	void InitDispatch(CaptureDispatch * pDispatch) { m_pDispatch = pDispatch; }
	/**
	 * Initializes capturing object. The parameter contains additional initialization
	 * data. The format of the data is device specific.
	 * @param sParams Start-up parameters of the device
	 * @return Operation success
	 */
	virtual ECaptureInitResult OpenDevice(const std::string sParams) = 0;
	/**
	 * Sets the path to an associated capture file. This call is used only by devices
	 * capable of saving traffic into a file. In the default case, the method
	 * doesn't cause any traffic to be saved.
	 * @param sFile Name of the target file or empty string if the traffic should not 
	 * be saved.
	 * @return \c true if the operation succeeds, \c false otherwise.
	 */
	virtual bool SetCaptureFile(const std::string sFile);
	/**
	 * Contains device-specific clean-up code.
	 */
	virtual void CloseDevice() = 0;	
	/**
	 * Sets up the capture filter. If the string is empty, no filtering should be applied.
	 * The format of the filtering parameter is device specific.
	 * @param sFilter Filtering data
	 * @return \c true if the operation was successful, \c false otherwise (e.g. in case
	 * of invalid filter format)
	 **/
	virtual bool SetFilter(const std::string sFilter) = 0;
	/**
	 * Starts or resumes data capture on the device
	 * @return \c true if the operation was successful, \c false otherwise
	 */	
	virtual bool StartCapture() = 0;
	/**
	 * Stops or suspends data capture on the device
	 * @return \c true if the operation was successful, \c false otherwise
	 */	
	virtual bool StopCapture() = 0;
	/**
	 * Returns the type of the capturing device.
	 * @return Capture type
	 */
	virtual ECaptureType GetCaptureType() = 0;
	/**
	 * Returns the offset of the time-based capturing filter. If the returned value is equal
	 * to zero, no time-based filter is applied.
	 * @return Offset used for time-based filtering
	 */
	struct timeval GetStartOffset() const { return m_offset; }
	/**
	 * Sets the offset of the time-based capturing filter. If the timestamp value is equal
	 * to zero, no time-based filter is applied. Otherwise the capturing device automatically
	 * discards captured messages whose timestamp is smaller than the specified offset. The
	 * offset is related to the first captured message.
	 * @param offset Offset used for time-based filtering
	 */
	void SetStartOffset(struct timeval offset);
};

#endif

