/**
 * @file GeneralConfigurationParams.h
 * Provides a class storage for TTCN-3 message GeneralConfigureReq
 * @author Yann garcia
 * @version 1.0
 * @date 24/07/2009
 */

#ifndef STF_370_GeneralConfigurationParams
#define STF_370_GeneralConfigurationParams

#include <boost/shared_ptr.hpp>
#include <string>
#include <vector>
#include <map>
#include <iostream>
#include <iomanip>

/*! Class EutIPInterface
 * \brief This class provides a storage for EUTS IP interface.
 */
class EutIPInterface
{
private:
  std::string m_eut;
  std::string m_eutAddress;
  int m_eutPort;
public:
  std::string& Eut() { return m_eut; };
  std::string& EutAddress() { return m_eutAddress; };
  int EutPort() { return m_eutPort; };
public:
  EutIPInterface(
    std::string eut,
    std::string eutAddress,
    int eutPort)
  {
    m_eut = eut;
    m_eutAddress = eutAddress;
    m_eutPort = eutPort;
  };
  ~EutIPInterface() { std::clog << "~EutIPInterface" << std::endl; };
 };
 
/*! Class GeneralConfigurationParams
 * \brief This class provides a class storage for TTCN-3 message GeneralConfigureReq.
 */
class GeneralConfigurationParams
{
private:
  std::string m_captureModuleAddress; /**< IP address of the TrafficCapture process */
  int m_captureModulePort; /**< Port number of the TrafficCapture listener */
  std::vector<std::string> m_ifaces; /**<  */
  unsigned char m_mode; /**< Capture mode: offline or online */
  int m_pcapTimeStampSeconds; /**< Pcap time stamps description */
  int m_pcapTimeStampMicroseconds; /**< Pcap time stamps description */
  unsigned char m_record; /**< Record mode (online mode only) */
  std::string m_pcapMergeTool; /**< Full path file name of the PCAP meger tool (offline mode only) */
  std::string m_fileToPlay; /**< PCAP file to play (offline mode only) */
  std::vector<std::string> m_mergerFileList; /**< List of the PCAP file to merge (offline mode only) */
  std::string m_mergerFilePath; /**< Directory path where the PCAP file to merge to merge are located (offline mode only) */
  std::vector<boost::shared_ptr<EutIPInterface> > m_eutIpInterfaceList; /**< List of the IP addresses/port numbers of the EUT equipments (online mode only) */
  
public:
  std::string& CaptureModuleAddress() { return m_captureModuleAddress; }; /**< Gets the IP address of the TrafficCapture process */
  int CaptureModulePort() { return m_captureModulePort; }; /**< Gets the port number of the TrafficCapture listener */
  std::vector<std::string>& Ifaces() { return m_ifaces; }; /**< Gets the list of the network interfaces to monitor */
  unsigned char Mode() { return m_mode; }; /**< Gets the capture mode */
  int PcapTimeStampSeconds() { return m_pcapTimeStampSeconds; }; /**< Gets the Pcap time stamps description */
  int PcapTimeStampMicroseconds() { return m_pcapTimeStampMicroseconds; }; /**< Gets the Pcap time stamps description */
  unsigned char Record() { return m_record; }; /**< Gets the record mode */
  std::string& PcapMergeTool() { return m_pcapMergeTool; }; /**< Gets the full path file name of the PCAP meger tool */
  std::string& FileToPlay() { return m_fileToPlay; }; /**< Gets the PCAP file to play */
  std::vector<std::string>& MergerFileList() { return m_mergerFileList; }; /**< Gets the list of the PCAP file to merge */
  std::string& MergerFilePath() { return m_mergerFilePath; }; /**< Gets the directory path where the PCAP file to merge to merge are located */
  std::vector<boost::shared_ptr<EutIPInterface> >& EutIpInterfaceList() { return m_eutIpInterfaceList; }; /**< Gets the list of the IP addresses/port numbers of the EUT equipments */
  
public:
  GeneralConfigurationParams() {};
  ~GeneralConfigurationParams() { std::clog << "~GeneralConfigurationParams" << std::endl; };
  
  void Initialize(
      std::string captureModuleAddress,
      int captureModulePort,
      std::vector<std::string> ifaces,
      unsigned char mode,
      unsigned char record,
      std::vector<boost::shared_ptr<EutIPInterface> > eutIpInterfaceList) {
    m_captureModuleAddress = captureModuleAddress;
    m_captureModulePort = captureModulePort;
    m_ifaces = ifaces;
    m_mode = mode;
    m_pcapTimeStampSeconds = 0;
    m_pcapTimeStampMicroseconds = 0;
    m_record = record;
    m_eutIpInterfaceList = eutIpInterfaceList;
  }; // ConnectToCaptureServerParams ctor
  void Initialize(
      std::string captureModuleAddress,
      int captureModulePort,
      unsigned char mode,
      int pcapTimeStampSeconds,
      int pcapTimeStampMicroseconds,
      std::string pcapMergeTool,
      std::string fileToPlay,
      std::vector<std::string>& mergerFileList,
      std::string mergerFilePath) {
    m_captureModuleAddress = captureModuleAddress;
    m_captureModulePort = captureModulePort;
    m_mode = mode;
    m_pcapTimeStampSeconds = pcapTimeStampSeconds;
    m_pcapTimeStampMicroseconds = pcapTimeStampMicroseconds;
    m_record = 0xff;
    m_pcapMergeTool = pcapMergeTool;
    m_fileToPlay = fileToPlay;
    m_mergerFileList = mergerFileList;
    m_mergerFilePath = mergerFilePath;
  }; // ConnectToCaptureServerParams ctor
};

#endif // STF_370_GeneralConfigurationParams
